import asyncio
import socket
import os
import fnmatch
import json
import time
import configparser
import traceback
import math
import re
import sys
import ast

from enum import IntEnum

from pylog.pylogger import PyLogger

from datetime import datetime

from py_pli.pylib import VUnits
from py_pli.pylib import send_msg
from py_pli.pylib import GlobalVar

import config_enum.scan_table_enum as scan_table_enum
import config_enum.excitationlight_selector_enum as els_enum
import config_enum.filter_module_slider_enum as fms_enum
import config_enum.bottom_light_director_enum as bld_enum
import config_enum.focus_mover_enum as fm_enum
import config_enum.platedoor_enum as pd_enum
import config_enum.detector_aperture_slider_enum as das_enum

from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal

from urpc_enum.measurementparameter import MeasurementParameter
from urpc_enum.serialparameter import SerialParameter

from fleming.common.firmware_util import EEFAnalogInput
from fleming.common.firmware_util import get_fan_control_endpoint
from fleming.common.firmware_util import get_node_endpoint
from fleming.common.firmware_util import get_serial_endpoint


class IntegratorMode(IntEnum):
    FULL_RESET  = 2
    LOW_RESET   = 3
    AUTO_RANGE  = 4
    FIXED_RANGE = 5
    LOW_RANGE   = 6
    HIGH_RANGE  = 7


class AnalogControl(IntEnum):
    FULL_OFFSET_RESET   = 4
    LOW_OFFSET_RESET    = 5
    HIGH_OFFSET_RESET   = 6
    SET_OFFSET          = 7


class fle_sys_meas_seq_generator(meas_seq_generator):

    def SetAnalogControl(self, aux=0, abs=0, ref=0, pmt2=0, pmt1=0):
        cmd = (0x0E000000 | ((aux & 7) << 18) | ((abs & 7) << 15) | ((ref & 7) << 12) | ((pmt2 & 7) << 3) | (pmt1 & 7))
        self.currSequence.append(cmd)

# Configuration Paths

ADJUSTMENT_PATH = '/home/pkiuser/.config/pyrunner/'
CONFIG_PATH = '/home/pkiuser/pyrunner/lib/python3.7/site-packages/PyRunner/config/'

# Hardware Mapping
# http://emeahama02:8080/jenkins/job/MLD/view/Fleming%20MAIN/job/3070D_Fleming_SourceCodeDoc/doxygen/page_fmbio.html
 
# mover-dict in order of homing
one_axis_mover = {
    'fm': VUnits.instance.hal.focusMover,
    'st': VUnits.instance.hal.scan_table,
    'pd': VUnits.instance.hal.plateDoor,
    'fms': VUnits.instance.hal.filterModuleSlider,
    'das1': VUnits.instance.hal.detectorApertureSlider1,
    'das2': VUnits.instance.hal.detectorApertureSlider2,
    'els': VUnits.instance.hal.excitationLightSelector,
    'bld': VUnits.instance.hal.bottomLightDirector,
    'sd': VUnits.instance.hal.doorLatch,
}


# Fans
# 2021-11-26/kstruebing
# node added
fans = {
    'base1' : {'node': 'fmb_fan','channel': 2}, # FAN 3 
    'base2' : {'node': 'fmb_fan','channel': 3}, # FAN 4 
    'base3' : {'node': 'fmb_fan','channel': 4}, # FAN 5 
#    'uslum' : {'node': 'fmb_fan','channel': 5}, # FAN 6 (not connected yet)
    'int'   : {'node': 'fmb_fan','channel': 0}, # FAN 1
    'ext'   : {'node': 'fmb_fan','channel': 1}, # FAN 2
    'pc'    : {'node': 'fmb_fan','channel': 6}, # FAN 7 (FMB Rear side)
    'fl'    : {'node': 'eef_fan','channel': 0}, # FAN 0 (connected to EEF)
    'trf'   : {'node': 'fmb_fan','channel': 7}, # FAN SW
}

# Temperature Sensors
# 2021-11-26/kstruebing
# conv added

temp_sensors = { # tmb = temp.-measurement-board
    'tmb_upper' : {'node': 'fmb', 'number': 0, 'conv': 256}, # Measurement chamber ceiling
    'tmb_lower' : {'node': 'fmb', 'number': 1, 'conv': 256}, # Measurement chamber bottom
    'tmb_in'    : {'node': 'fmb', 'number': 2, 'conv': 256}, # Airventilation unit inlet to internal fan
    'tmb_out'   : {'node': 'fmb', 'number': 3, 'conv': 256}, # Airventilation unit outlet
#    'tmb_rt'    : {'node': 'fmb', 'number': 4}, # Room Temperature at Instrument inlet (bottom)
    'tmb_fl'    : {'node': 'fmb', 'number': 5, 'conv': 256}, # Flashlamp Fan
    'tmb_int'   : {'node': 'fmb', 'number': 6, 'conv': 256}, # AirV. Peltier Cold side
    'tmb_ext'   : {'node': 'fmb', 'number': 7, 'conv': 256}, # AirV. Peltier Hot side
    'tmb_am'    : {'node': 'fmb', 'number': 16, 'conv': 175}, # Ambient Temperature Sensor
    'tmb_hum'   : {'node': 'fmb', 'number': 17, 'conv': 100}, # Humidity Sensor
#    'tmb_p1'    : {'node': 'eef', 'number': 9, 'conv': 256},   # PMT 1 Temp, EEF Rev. 3
    'tmb_p1'    : {'node': 'eef', 'number': 15, 'conv': 256},   # PMT 1 Temp, EEF Rev. 4
#    'tmb_p2'    : {'node': 'eef', 'number': 10, 'conv': 256},  # PMT 2 Temp, EEF Rev. 3
    'tmb_p2'    : {'node': 'eef', 'number': 16, 'conv': 256},  # PMT 2 Temp, EEF Rev. 4
}

# Temperature-Elements
# FLE-1920 2022-07-06
temp_elements = {
    'htu'   : 0, # Upper heating
    'htl'   : 1, # Lower heating
    'htb'   : 2, # Dispenser bottle heating
    'tec'   : 3, # 
#    'av2_2' : 4, # AVU Peltier 2, Pin 2
#    'av2_1' : 5, # AVU Peltier 2, Pin 1
#    'av1_2' : 6, # AVU Peltier 1, Pin 2
#    'av1_1' : 7, # AVU Peltier 1, Pin 1
    'av1'   : 15, # AVU Peltier Pin 1
    'av2'   : 16, # AVU Peltier Pin 2
    'pmt_1' : 9, # PMT 1 Peltier
    'pmt_2' : 10, # PMT 1 Peltier
}

pwm_status = {
    'AV_otw'   : 11, # PWM 2 Driver (AV1+ 2) Overtemperature Warning active low
    'AV_fault' : 10, # PWM 2 Driver (AV1+ 2) Fault active low
    'HT_otw'   : 9, # PWM 1 Driver (HTL+ U) Overtemperature Warning active low
    'HT_fault' : 8, # PWM 1 Driver (HTL+ U) Fault active low
    'PMT_AL_otw'  : 25, # PWM driver PMT1-TEC, PMT2-TEC and Alpha Laser TEC overtemperature warning (active low)
    'PMT_AL_fault': 24 # PWM driver PMT1-TEC, PMT2-TEC and Alpha Laser TEC fault signal (active low)
}

alpha_feedback = {
    'alphaTempError'  : 0, # Alpha Laser Overtemp Error (Digital Input EEF)
    'alphaPhotoDiode' : 2, # Alpha Laser Internal Photodiode (Analog Input EEF)
    'alphaTempIn'     : 3, # Alpha Laser Temperature Sensor (Analog Input EEF)
}

# Helper Functions

def create_function_list(prefix =""):
    '''
    Creates a text-file with a list of all function-names of the current script
    Usefull to put them into servicetasks.adj
    '''
    script = f"{sys._getframe().f_code.co_filename}"
    with open(script, 'r') as file:
        tree = ast.parse(file.read(), filename=script)

    function_names = [node.name for node in ast.walk(tree) if isinstance(node, ast.FunctionDef)]   
    list_name = script.replace(".py", ".fun")

    with open(list_name, 'w') as file:
        for f in function_names:
            file.write(prefix + f + '\n')
    
    return(list_name)


# Configuration File Handling

def get_adjust_value(adjust_file, parameter_name):
    ''' Get the value of a parameter from the given adjustment file
    '''
    
    adjust = configparser.ConfigParser()
    adjust.read(adjust_file)
    for section in adjust.sections():
        if parameter_name in adjust[section]:
            return adjust[section][parameter_name]
        else:
            return None

def get_config_value(unit, parameter_name):
    ''' Get the value of a parameter from the given config file
    '''
    adj_name = configparser.ConfigParser()
    adj_file = f'{ADJUSTMENT_PATH}{unit}.adj'
    adj_name.read(adj_file)
    for section in adj_file.sections():
        # Look in Adjustment File
        if parameter_name in adj_file[section]:
            return adj_file[section][parameter_name]
        else:
            # Look in Configuration File
            cfg_name = configparser.ConfigParser()
            cfg_file = f'{CONFIG_PATH}{unit}.cfg'
            cfg_name.read(cfg_file)
            if parameter_name in cfg_name[section]:
                return cfg_name[section][parameter_name]
            else:
                return(f'Parameter {parameter_name} not found or {unit} invalid')

def get_config_files():
    files = fnmatch.filter(os.listdir(CONFIG_PATH), '*.cfg')
    return(files)

def get_adjustment_files():
    files = fnmatch.filter(os.listdir(ADJUSTMENT_PATH), '*.adj')
    return(files)


def py_ver():
    section = 'Application'
    filename = '/home/pkiuser/pyrunner/lib/python3.7/site-packages/PyRunner/config/HAL.cfg'
    hal = configparser.ConfigParser()
    hal.read(filename)
    return hal[section]['Version']

# Data Aquisition #############################################################

# Mitutoyo Mechanical Gauge
def setup_serial_mitutoyo():
    serial_port = serial.Serial(port = SERIAL_PORT, baudrate=BAUDRATE, bytesize=BYTESIZE, timeout=2, parity=PARITY, stopbits=STOPBITS)
    return (serial_port)

def read_value(sensor):
    sensor.write(b'GA01\r\n') # Read display value
    value = float(sensor.read(size=20)[5:15])
    return(value)

def current_value(sensor):
    sensor.write(b'CN01\r\n') # Set display to current value
    return(1)

def max_value(sensor):
    sensor.write(b'CX01\r\n') #set display to max value
    return(1)

def set_zero(sensor):
    sensor.write(b'CR01\r\n') #set display to zero
    response = sensor.read(size=20)
    return(response)


def get_test_config(testname):
    filename = f'./{testname}.cfg'
    test_cfg = configparser.ConfigParser()
    test_cfg.read(filename)
    return(test_cfg)

def get_reportfile(path, base_name, test):
# Returns the next reportfile name for a test function
# Created 2021-09-03 10:21 
# Checked
#  Tested

    instrument = socket.gethostname()
    today = datetime.now().strftime('%Y%m%d')
    reportpath = f'{path}/{instrument}_{today}'
    if not os.path.isdir(reportpath):
        os.mkdir(reportpath)
    reportfile = f'{base_name}_{test}_Run'
    no = len(fnmatch.filter(os.listdir(reportpath), reportfile + '*.md')) + 1
    reportfile = f'{reportfile}{no:03d}.md'
    return(f'{reportpath}/{reportfile}')

def write_report_header(filename, user = 'no_one', comment =''):
    create_time = datetime.now().strftime('%H:%M:%S')
    header = f"\n Report {filename} created {create_time} by {user}\n Comment: {comment}\n"
#    report_header_template = '/home/pkiuser/hardware_scripts/fleming/system_test/report_header.md'
    with open(filename, 'a') as rep_file:
        rep_file.write(header)
    return header

def write_report_record(filename, record):
    with open(filename, 'a') as rep_file:
        rep_file.write(record + '\n')
    return

def get_datafile(path, base_name, test_function):
# Returns the next datafile name for a test function
# Created 2021-09-03 10:21 
# Checked
#  Tested

    instrument = socket.gethostname()
    today = datetime.now().strftime('%Y%m%d')
    datapath = f'{path}/{instrument}_{today}'
    if not os.path.isdir(datapath):
        os.mkdir(datapath)
    datafile = f'{base_name}_{test_function}_Run'
    no = len(fnmatch.filter(os.listdir(datapath), datafile + '*.csv')) + 1
    datafile = f'{datafile}{no:03d}.csv'
    return(f'{datapath}/{datafile}')

def write_test_header(file_name, parameter):
    start_time = datetime.now().strftime('%H:%M:%S')
    header = f"{file_name} started at {start_time}\n"

    for p in parameter:
        header += f"{p}: {parameter[p]}\n"

    columns = 'sample;time;'
    for t in temp_sensors:
        columns += f'{t};'
    columns += '\n'

    header += columns

    with open(file_name, 'a') as data:
        data.write(header)
    return header   


def bargraph(x, x_max, res = 80, char = '|'):
    return(char.ljust(int(x / x_max * res), char))


# Version of Hardware

async def hw_ver(time_out = 1):
    # board = eef, fmb, MC6
    # fw_index = 0: Bootloader; 1: Firmware; 2: FPGA (only EEF)
    boards = {
        'eef': 3,
        'fmb': 2,
        'MC6': 2,
    }
    versions = {}
    for b in boards:
        for i in range(boards[b]):
            version = await VUnits.instance.hal.nodes[b].endpoint.GetVersion(i, time_out)
            info = f'{b}      : {version}'
            print(info)
            await send_msg(json.dumps({'result': info}))
            versions[f'{b}_{i}'] = version
    return versions

# Writes msg to Logger and GC Output
# Created 2021-09-03 10:21
# Checked
#  Tested    
async def talk(test_name, msg):
    message = f'{test_name}: {msg}'
    PyLogger.logger.info(message)
    await send_msg(json.dumps({'result': message}))

async def gc_out(test_name, msg):
    message = f'{test_name}: {msg}'
    await send_msg(json.dumps({'result': message}))


# Door Lock
async def init_door(switch_current = 1000, hold_current = 100, switch_time = 100):
    # switch_current : 1000 = 24VDC (100% Duty Cycle)
    # hold_current: dto
    # switch_time: time for switch_current, then hold_current (ms)
    # switch_current = 0: switch_current ignored

    pwm1 = 0
    mc6 = VUnits.instance.hal.nodes['MC6'].node
    await mc6.EnablePWMOutput(pwm1, 0)
    await mc6.ConfigurePWMOutput(pwm1, switch_current, hold_current, switch_time)
    return('Door Solenoid initialized')

async def door(lock = 0):
    pwm1 = 0
    mc6 = VUnits.instance.hal.nodes['MC6'].node    
    await mc6.EnablePWMOutput(pwm1, lock)
    return(f'Door Solenoid PWM = {lock}')   

async def door_lock():
    await door(1)
    return(f'Door locked')   

async def door_unlock():
    await door(0)
    return(f'Door unlocked')   

async def sliding_door_test(cycles = 10, delay = 0.5, switch_current = 1000, hold_current = 100, switch_time = 100):
    TEST_NAME = 'sliding_door_test'
    await init_door(switch_current, hold_current, switch_time)
    switch = 1
    for i in range(cycles):  
        message = f'Cycle: {i + 1}/{cycles}'
        await talk(TEST_NAME, message)
        await door(switch)
        await asyncio.sleep(0.5)
        switch ^= 1
        await door(switch)
        await asyncio.sleep(delay)        
        switch ^= 1        
    return(f'{TEST_NAME} Done.')   

# Flashlamp Functions #########################################################

#            Copied from fleming.module_test.ex_test

async def fl_test(duration_s=1, frequency=100, power=0.0):
    """
    Switch on Flashlamp for x secs.

    duration_s: The duration in seconds.
    frequency: The flash frequency in Hz.
    power: The flash power in the range [0.0, 1.0].
    """
    TEST_NAME = sys._getframe().f_code.co_name

    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    flashes_corse, flashes_fine = divmod((duration_s * frequency), 65536)
    duty_cycle = 0.5
    ontime_us  = int(1 / frequency * 1e6 * duty_cycle)
    offtime_us = int(1 / frequency * 1e6 * (1 - duty_cycle))
    
    await send_msg(json.dumps({'result': f"Flash-Lamp Check, Duration {duration_s} sec, Frequency {frequency} Hz, Power {power}"}))

    # Switch on Flashlamp Fan
    await fan_pwm('fl', 100)
    # await fl_fan(100)
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0, timeout=1)
    await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.FlashLampPower, power, timeout=1)

    op_id = TEST_NAME
    seq_gen = meas_seq_generator()
    if flashes_corse > 0:
        seq_gen.Loop(flashes_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(ontime_us * 100)
        seq_gen.SetSignals(OutputSignal.Flash)
        seq_gen.TimerWaitAndRestart(offtime_us * 100)
        seq_gen.ResetSignals(OutputSignal.Flash)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if flashes_fine > 0:
        seq_gen.Loop(flashes_corse)
        seq_gen.TimerWaitAndRestart(ontime_us * 100)
        seq_gen.SetSignals(OutputSignal.Flash)
        seq_gen.TimerWaitAndRestart(offtime_us * 100)
        seq_gen.ResetSignals(OutputSignal.Flash)
        seq_gen.LoopEnd()
    seq_gen.Stop(0)

    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    await asyncio.sleep(duration_s)
    await fan_pwm('fl', 0)
    #await fl_fan(0)
    return f"{TEST_NAME} done"

# Alpha Functions #############################################################

# Updated by rbartz

async def al_init(filter = 0):
    # Instrument initialized
    # Initialize alpha beampath
    # ELS Alpha=245.8
    
    els = VUnits.instance.hal.excitationLightSelector
    fms = VUnits.instance.hal.filterModuleSlider
    els.GotoPosition(els_enum.Positions.Alpha)
    if filter > 0:
        filter_position = fms_enum.Position.IndexZeroPosMeas + (filter * fms_enum.GC_Params.IndexOffset_c)
    else:
        filter_position = fms_enum.Position.FixMirrorPosition_c
    fms.Move(filter_position)
    
    await al_enable()
    await al_settemp()
    await al_setpower()
    
    return 'Alpha Laser Initialized'


async def al_enable():
    measurement_unit = VUnits.instance.hal.measurementUnit
    # await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.AlphaLaserEnable, 1, timeout=1)
    # Teemu 2022-06-27
    await measurement_unit.endpoint.SetParameter(MeasurementParameter.AlphaLaserEnable, 1, timeout=1)    
    await asyncio.sleep(0.1)


async def al_disable():
    measurement_unit = VUnits.instance.hal.measurementUnit
    # Teemu 2022-06-27
    # await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.AlphaLaserEnable, 0, timeout=1)
    await measurement_unit.endpoint.SetParameter(MeasurementParameter.AlphaLaserEnable, 0, timeout=1)
    await asyncio.sleep(0.1)       


async def al_setpower(power=0.47):
    measurement_unit = VUnits.instance.hal.measurementUnit
    # Teemu 2022-06-27    
    # await measurement_unit.MeasurementFunctions.SetParameter(MeasurementParameter.AlphaLaserPower, power, timeout=1)
    await measurement_unit.endpoint.SetParameter(MeasurementParameter.AlphaLaserPower, power, timeout=1)
    await asyncio.sleep(0.1)


async def al_settemp(tref=0.38):  # 0.33
    temp_control_unit = VUnits.instance.hal.alphaLaserStandard_Cooling
    await temp_control_unit.set_target_temperature(tref)


async def al_gettemp():
    temp_control_unit = VUnits.instance.hal.alphaLaserStandard_Cooling
    return (await temp_control_unit.get_feedback_value())


async def al_getpd():
    eef = VUnits.instance.hal.nodes['eef']
    return (await eef.GetAnalogInput(EEFAnalogInput.ALPHAPHOTODIODE, timeout=1))[0]


async def al_on():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'al_on'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.Alpha)    # Reset Signal to turn Laser On!!!!
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    PyLogger.logger.info('=============Alpha-Laser On!!!==================')


async def al_off():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'al_off'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.Alpha)    # Set Signal to turn Laser Off!!!
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    PyLogger.logger.info('=============Alpha-Laser Off!!!==================')

async def write_alpha(filename, sample):
    # Write Alpha Laser Temp Data to file

    row = f'{sample:04.0f};'
    row += f'{await al_gettemp()};'
    row += f'{await al_getpd()};'
    row += '\n'
    with open(filename, 'a') as file:
        file.write(row)
    return()

# Temperature Control Alpha Laser
# .vu.hal.alphaLaserStandard_Cooling.endpoint.Configure(channel=0, dt=0.1, kp=0.2, ti=0.3, td=0.4, min=0.5, max=0.6, timeout=1)


# Photodiode ABS/Ref Functions ################################################

# Updated by rbartz

async def ref_test_v0(iterations=1, exposuretime_us=100, delay=1):
    '''
    rb 210927:
    Test reference- and absorbance-photodiode.
    This function uses the basic integrator reset and not the integrator mode.
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    exptime = exposuretime_us * 100
    resettime = 100000  # 1 ms

    op_id = 'ref_test_v0'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)
    seq_gen.TimerWaitAndRestart(resettime)
    seq_gen.SetSignals(OutputSignal.IntRstAbs | OutputSignal.IntRstRef)
    seq_gen.TimerWaitAndRestart(exptime)
    seq_gen.ResetSignals(OutputSignal.IntRstAbs | OutputSignal.IntRstRef)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.SetSignals(OutputSignal.IntRstAbs | OutputSignal.IntRstRef)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 4)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] * 4
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
            msg = f"ref_test_v0() failed: {ex}"
            PyLogger.logger.error(msg)
            await send_msg(json.dumps({'result': msg}))

    return resultsum
    

async def ref_test_v7(iterations=1, exposuretime_us=100, delay=1):
    '''
    rb 210927:
    Test reference- and absorbance-photodiode.
    This function uses the integrator mode and offset correction for the PDD board XIQ02-V7.
    exposuretime_us must be greater or equal to 42 us !!!
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    exptime = exposuretime_us * 100
    reset_delay      =  40000  # 400 us
    conversion_delay =   1200  #  12 us
    switch_delay     =     25  # 250 ns
    fixed_range      =   2000  #  20 µs

    op_id = 'ref_test_v7'
    seq_gen = fle_sys_meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)
    seq_gen.SetAnalogControl(abs=AnalogControl.FULL_OFFSET_RESET, ref=AnalogControl.FULL_OFFSET_RESET)
    # Enable integrator full reset for min 400 us
    seq_gen.TimerWaitAndRestart(reset_delay)
    seq_gen.SetIntegratorMode(abs=IntegratorMode.FULL_RESET, ref=IntegratorMode.FULL_RESET)
    # Measure high range offset. The ADC conversion requires 12 us.
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleAbs | TriggerSignal.SampleRef)
    # Start the integrator in low range with high reset. The conversion is done now, so we can set the high range offset register.
    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(abs=IntegratorMode.LOW_RESET, ref=IntegratorMode.LOW_RESET)
    seq_gen.SetAnalogControl(abs=AnalogControl.SET_OFFSET, ref=AnalogControl.SET_OFFSET)
    # Measure the low range offset and switch the inegrator to auto range mode. (This is the start of the measurement window.)
    seq_gen.TimerWaitAndRestart(exptime - fixed_range)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleAbs | TriggerSignal.SampleRef)
    seq_gen.SetIntegratorMode(abs=IntegratorMode.AUTO_RANGE, ref=IntegratorMode.AUTO_RANGE)
    # Switch the integrator to fixed range mode. The conversion is done noe, so we can set the low range offset regster.
    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetIntegratorMode(abs=IntegratorMode.FIXED_RANGE, ref=IntegratorMode.FIXED_RANGE)
    seq_gen.SetAnalogControl(abs=AnalogControl.SET_OFFSET, ref=AnalogControl.SET_OFFSET)
    # Measure and save the analog result.
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=3)
    # Enable integrator full reset.
    seq_gen.SetIntegratorMode(abs=IntegratorMode.FULL_RESET, ref=IntegratorMode.FULL_RESET)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 4)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] * 4
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
            msg = f"ref_test_v7() failed: {ex}"
            PyLogger.logger.error(msg)
            await send_msg(json.dumps({'result': msg}))

    return resultsum


# PMT Functions ###############################################################

# Updated by rbartz
# Low Level
async def hvpmt1on():
    # Reset Signal to turn on!!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvpmt1on'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.HVOnPMT1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    PyLogger.logger.info('=============HV PMT1 On!!!==================')


async def hvpmt2on():
    # Reset Signal to turn on!!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvpmt2on'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.HVOnPMT2)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    PyLogger.logger.info('=============HV PMT2 On!!!==================')


async def hvpmt1off():
    #Set Signal to turn off!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvpmt1off'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.HVOnPMT1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    PyLogger.logger.info('=============HV PMT1 Off!!!==================')


async def hvpmt2off():
    #Set Signal to turn off!!
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvpmt2off'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.HVOnPMT2)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)
    PyLogger.logger.info('=============HV PMT2 Off!!!==================')


async def hvgate1on():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvgate1on'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.HVGatePMT1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)


async def hvgate2on():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvgate2on'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OutputSignal.HVGatePMT2)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)


async def hvgate1off():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvgate1off'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.HVGatePMT1)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)


async def hvgate2off():
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'hvgate2off'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OutputSignal.HVGatePMT2)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)

# High level



# Temperature Control Functions ###############################################

def get_PID(unit):
    # Returns Kp, Ti, Td Values from Adjustmentfile
    filename = f'{ADJUSTMENT_PATH}{unit}'
    pid = {
        'Kp' : 0.0, 
        'Ti' : 1.0,
        'Td' : 0.0,
    }
    for p in pid:
        pid[p] = float(get_adjust_value(filename,p))
    return pid    

# Fan Control
# [x] tested 2021-03-12/KS


async def fan_pwm(fan_name, pwr):
    # 2021-11-26/kstruebing
    # node added

    """
    fan_name: base1, base2, base3, int, ext, pc, fl, trf
    pwr: 0...100%
    """
    if (pwr < 0) or (pwr > 100):
        raise ValueError(f"pwr must be in the range [0, 100]")

    fan = get_fan_control_endpoint(fans[fan_name]['node'])
    channel = fans[fan_name]['channel']
    await fan.SetSpeed(channel, pwr, timeout=1)
    await fan.Enable(channel, (pwr > 0), timeout=1)
    return f"Fan '{fan_name}' @ {int(pwr)}%"

async def set_base_fans_default(pwr = 10):
    # 2021-12-08/kstruebing
    base_fans = [2, 3, 4]
    node = 'eef_fan'
    if (pwr < 0) or (pwr > 100):
        raise ValueError(f"pwr must be in the range [0, 100]")

    fan = get_fan_control_endpoint(node)
    for ch in base_fans:
        await fan.SetSpeed(ch, pwr, timeout=1)
        await fan.Enable(ch, (pwr > 0), timeout=1)
    return

async def set_base_fans(pwr = 20):
    # 2023-01-25/kstruebing
    base_fans = ["base1", "base2", "base3"]
    timeout = 1

    if (pwr < 0) or (pwr > 100):
        raise ValueError(f"pwr must be in the range [0, 100]")

    for f in base_fans:
        fan = get_fan_control_endpoint(fans[f]['node'])
        ch = fans[f]['channel']
        await fan.SetSpeed(ch, pwr, timeout)
        await fan.Enable(ch, (pwr > 0), timeout)

    return(f"Base Fans set to {pwr}%")


async def fans_off():
    # 2021-11-26/kstruebing
    # node added
    timeout = 1
    pwr = 0
#    for ch in range(8):
#        fan = get_fan_control_endpoint(fans[fan_name]['node'])
#        await fan.SetSpeed(ch, pwr, timeout)
#        await fan.Enable(ch, (pwr > 0), timeout)

    for f in fans:
        fan = get_fan_control_endpoint(fans[f]['node'])
        ch = fans[f]['channel']
        await fan.SetSpeed(ch, pwr, timeout)
        await fan.Enable(ch, (pwr > 0), timeout)

    return f"All fans off"


# [x] tested 2021-04-01/KS
async def fan_sw(fan_name, sw):
    # 2021-11-26/kstruebing
    # node added
    """
    fan_name: base1, base2, base3, int, ext, pc, fl, trf
    sw: 0, 1
    """
    if (sw != 0) and (sw != 1):
        raise ValueError(f"sw must be 0 or 1")

    fan = get_fan_control_endpoint(fans[fan_name]['node'])

    channel = fans[fan_name]['channel']
    await fan.SetSpeed(channel, 100, timeout=1)
    await fan.Enable(channel, sw, timeout=1)
    return f"Fan '{fan_name}' enable = {sw}"

# Temp.-Sensors

# [] tested 
async def get_tmp(tmb_name):
    # 2021-11-26/kstruebing
    # Conversion factor added
    """
    TMB: Readout of single TMB-Sensor
    tmb_name:
    tmb_upper: Upper Heating
    tmb_lower: Lower Heating
    tmb_in:    AVU internal Fan inlet 
    tmb_out:   AVU internal Fan outlet
    tmb_fl:    Flashlamp Temp
    tmb_int:   AVU Internal Fan (Cool side Peltier) 
    tmb_ext:   AVU external Fan (Hot side Peltier)
    tmb_am:    Ambient Temp
    tmb_hum:   Humidity Sensor
    tmb_p1:    PMT 1  
    tmb_p2:    PMT 2
    """
    if tmb_name not in temp_sensors:
        raise ValueError(f"{tmb_name} not valid.")
    
    node = get_node_endpoint(temp_sensors[tmb_name]['node'])
    value = (await node.GetAnalogInput(temp_sensors[tmb_name]['number']))[0] * int(temp_sensors[tmb_name]['conv'])
    return value

# [] tested 
async def get_pwm_status(signal = 'AV_fault', board = 'fmb'):
    # 2022-01-04/kstruebing
    # FLE-1722
    """
    signal: Readout of PWM Status
    'AV_otw'      : 11, # PWM 2 Driver (AV1+ 2) Overtemperature Warning
    'AV_fault'    : 10, # PWM 2 Driver (AV1+ 2) Fault
    'HT_otw'      : 9, # PWM 1 Driver (HTL+ U) Overtemperature Warning
    'HT_fault'    : 8, # PWM 1 Driver (HTL+ U) Fault
    'PMT_AL_otw'  : 25, # PWM driver PMT1-TEC, PMT2-TEC and Alpha Laser TEC overtemperature warning (active low)
    'PMT_AL_fault': 24 # PWM driver PMT1-TEC, PMT2-TEC and Alpha Laser TEC fault signal (active low)
    
    """
    if signal not in pwm_status:
        raise ValueError(f"{signal} not valid.")
    
    control = get_node_endpoint(board)
    value = (await control.GetDigitalInput(pwm_status[signal]))[0]
    return value


async def get_all_tmp():
    """
    Returns all temp-values in one string
    
    TMB: Readout of all TMB-Sensor
    tmb_name:
    tmb_upper: Upper Heating
    tmb_lower: Lower Heating
    tmb_in:    AVU internal Fan inlet 
    tmb_out:   AVU internal Fan outlet
    tmb_rt:    Ambient Temp
    tmb_fl:    Flashlamp Temp
    tmb_int:   AVU Internal Fan (Cool side Peltier)
    tmb_am:    Ambient Temp
    tmb_hum:   Humidity Sensor
    tmb_ext:   AVU external Fan (Hot side Peltier)
    tmb_p1:    PMT 1  
    tmb_p2:    PMT 2
    """
    result = {}
    for tmb in temp_sensors:
        value = await get_tmp(tmb)
        result[tmb] = float(f'{value:.2f}')
        message = f'{tmb}     : {value:.2f}'
        PyLogger.logger.info(message)
        await send_msg(json.dumps({'result': message}))
    return result

# writes the header-line for data aquisitions of temperatures
# Created 2021-09-03 10:21 
# Checked
#  Tested
# Last change 2022-01-04
# PWM Status added
def write_temp_header(file_name):
    str = 'sample;time;'
    for t in temp_sensors:
        str += f'{t};'
    for s in pwm_status:
        str += f'{s};'
    str += '\n'
    start_time = datetime.now().strftime('%H:%M:%S')
    with open(file_name, 'a') as data:
        data.write(f"{file_name} started at {start_time}\n")
        data.write(str)
    return str   

async def write_temp(filename, sample):
    # Reads Temp-Values and writes in filename
    # Created 2021-09-03 10:21
    # Checked
    #  Tested
    # Last change 2022-01-04
    # PWM Status added

    time_s = datetime.now().strftime('%H:%M:%S')
    row = f'{sample:04.0f};{time_s};'
    for t in temp_sensors:
        row += f'{await get_tmp(t):.2f};'
#   Status of PWM Driver added for FLE-1722
    for s in pwm_status:
        row += f'{await get_pwm_status(s)};'
    row += '\n'
    with open(filename, 'a') as data:
        data.write(row)
    return row

async def write_temp_offset(filename, ref_temp = 25.0):
# Calculate offset of all temperature values against one ref-value
# Created 2021-10-01 
# Checked
#  Tested

    time_s = datetime.now().strftime('%H:%M:%S')
    sample = 0
    row = f'{sample:04.0f};{time_s};'
    for t in temp_sensors:
        row += f'{(await get_tmp(t) - ref_temp):.2f};'
    row += '\n'
    with open(filename, 'a') as data:
        data.write(row)
    return row

# Temperature Elements
# Turns Peltiers of airventilation unit ON
# 2022-06-14/kstruebing AVU TEC rework
# [] tested 

async def tec_on(pwm = 65, fan = 10):
    min_pwm = 0.05
    fmb = VUnits.instance.hal.nodes['fmb']
    await fan_pwm('int', fan)
    await fan_pwm('ext', 100)
#    p_11 = temp_elements['av1_1']
#    p_12 = temp_elements['av1_2']
#    p_21 = temp_elements['av2_1']
#    p_22 = temp_elements['av2_2']
    p_1 = temp_elements['av1']
    p_2 = temp_elements['av2']

#    await fmb.SetAnalogOutput(p_11, 0.0)
#    await fmb.SetAnalogOutput(p_12, (pwm / 100))
#    await fmb.SetAnalogOutput(p_21, 0.0)
#    await fmb.SetAnalogOutput(p_22, (pwm / 100))
    await fmb.SetAnalogOutput(p_1, (min_pwm))
    await fmb.SetAnalogOutput(p_2, (pwm / 100) + min_pwm)

    return('AVU TE switched on')

# Turns Peltiers of airventilation unit OFF
# [] tested     
async def tec_off():
    fmb = VUnits.instance.hal.nodes['fmb']
    min_pwm = 0.05
#    p_11 = temp_elements['av1_1']
#    p_12 = temp_elements['av1_2']
#    p_21 = temp_elements['av2_1']
#    p_22 = temp_elements['av2_2']    
    p_1 = temp_elements['av1']
    p_2 = temp_elements['av2']

#    await fmb.SetAnalogOutput(p_11, 0.0)
#    await fmb.SetAnalogOutput(p_12, 0.0)
#    await fmb.SetAnalogOutput(p_21, 0.0)
#    await fmb.SetAnalogOutput(p_22, 0.0)
    await fmb.SetAnalogOutput(p_1, min_pwm)
    await fmb.SetAnalogOutput(p_2, min_pwm)

    return('AVU TEC switched off.')
    
async def h_on(lower_pwm = 96, upper_pwm = 96, fan = 10):
    # Turns Heating foils ON
    # [] tested     

    fmb = VUnits.instance.hal.nodes['fmb']
    h_lower = temp_elements['htl']
    h_upper = temp_elements['htu']
    await fan_pwm('int', fan)
    await fmb.SetAnalogOutput(h_lower, lower_pwm / 100.0)
    await fmb.SetAnalogOutput(h_upper, upper_pwm / 100.0)
    return('Lower, Upper Heating switched ON.')

async def h_off():
    # Turns Heating foils OFF
    # [] tested          

    fmb = VUnits.instance.hal.nodes['fmb']
    h_lower = temp_elements['htl']
    h_upper = temp_elements['htu']
    await fmb.SetAnalogOutput(h_lower, 0.0)
    await fmb.SetAnalogOutput(h_upper, 0.0)
    return('Lower, Upper Heating switched OFF.')

async def p_on(ch = 1, pwm = 30):
    # PMT Cooling ON
    # [] tested     

    eef = VUnits.instance.hal.nodes['eef']
    if ch == 1:
        await eef.SetAnalogOutput(temp_elements['pmt_1'], (pwm / 100))
    else:
        await eef.SetAnalogOutput(temp_elements['pmt_2'], (pwm / 100))
    
async def p_off(ch = 1):
    eef = VUnits.instance.hal.nodes['eef']
    if ch == 1:
        await eef.SetAnalogOutput(temp_elements['pmt_1'], 0.0)
    else:
        await eef.SetAnalogOutput(temp_elements['pmt_2'], 0.0)

async def e_off():
    await h_off()
    await tec_off()
    return('All Temperature Elements switched OFF.')
   

